/*******************************************************************************
 * Copyright (c) 2014, 2019 Ericsson
 *
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Vincent Perot - Initial API and implementation
 *   Viet-Hung Phan - Support pcapNg
 *******************************************************************************/

package org.eclipse.tracecompass.pcap.core.tests.file;

import static org.junit.Assert.assertEquals;
import static org.junit.Assume.assumeTrue;

import java.io.IOException;
import java.nio.ByteOrder;

import org.eclipse.tracecompass.internal.pcap.core.trace.BadPcapFileException;
import org.eclipse.tracecompass.internal.pcap.core.trace.PcapOldFile;
import org.eclipse.tracecompass.pcap.core.tests.shared.PcapTestTrace;
import org.junit.Test;

/**
 * JUnit Class that tests the opening of valid pcap files.
 *
 * @author Vincent Perot
 */

public class PcapFileOpenTest {

    /**
     * Test that verify that an empty pcap or pcapNg file is properly opened and that the
     * file properties are correct.
     *
     * @throws BadPcapFileException
     *             Thrown when the file is erroneous. Fails the test.
     * @throws IOException
     *             Thrown when an IO error occurs. Fails the test.
     */
    @Test
    public void FileOpenEmptyTest() throws IOException, BadPcapFileException {

        PcapTestTrace trace = PcapTestTrace.EMPTY_PCAP;
        assumeTrue(trace.exists());
        // Get a right pcap/pcapNg trace
        try (PcapOldFile file = (PcapOldFile) trace.getTrace()) {
            assertEquals(PcapTestTrace.EMPTY_PCAP.getPath(), file.getPath());
            assertEquals(2, file.getMajorVersion());
            assertEquals(4, file.getMinorVersion());
            assertEquals(1, file.getDataLinkType());
            assertEquals(65535, file.getSnapShotLength());
            assertEquals(0, file.getTimeAccuracy());
            assertEquals(0, file.getTimeZoneCorrection());
            assertEquals(ByteOrder.LITTLE_ENDIAN, file.getByteOrder());

            assertEquals(0, file.getTotalNbPackets());
        }

    }

    /**
     * Test that verify that an non-empty pcap file is properly opened and that
     * the file properties are correct.
     *
     * @throws BadPcapFileException
     *             Thrown when the file is erroneous. Fails the test.
     * @throws IOException
     *             Thrown when an IO error occurs. Fails the test.
     */

    @Test
    public void FileOpenTest() throws IOException, BadPcapFileException {

        PcapTestTrace trace = PcapTestTrace.MOSTLY_TCP;
        assumeTrue(trace.exists());
        // Get a right pcap/pcapNg trace
        try (PcapOldFile file = (PcapOldFile) trace.getTrace();) {
            assertEquals(PcapTestTrace.MOSTLY_TCP.getPath(), file.getPath());
            assertEquals(2, file.getMajorVersion());
            assertEquals(4, file.getMinorVersion());
            assertEquals(1, file.getDataLinkType());
            assertEquals(65535, file.getSnapShotLength());
            assertEquals(0, file.getTimeAccuracy());
            assertEquals(0, file.getTimeZoneCorrection());
            assertEquals(ByteOrder.LITTLE_ENDIAN, file.getByteOrder());

            assertEquals(43, file.getTotalNbPackets());
        }
    }
}
