/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import java.util.List;
import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;

/**
 * Capabilities specific to {@link WorkspaceEdit}s
 */
@SuppressWarnings("all")
public class WorkspaceEditCapabilities {
  /**
   * The client supports versioned document changes in {@link WorkspaceEdit}s
   */
  private Boolean documentChanges;

  /**
   * The resource operations the client supports. Clients should at least
   * support 'create', 'rename' and 'delete' files and folders.
   * <p>
   * See {@link ResourceOperationKind} for allowed values.
   */
  @ProtocolSince("3.13.0")
  private List<String> resourceOperations;

  /**
   * The failure handling strategy of a client if applying the workspace edit
   * fails.
   * <p>
   * See {@link FailureHandlingKind} for allowed values.
   */
  @ProtocolSince("3.13.0")
  private String failureHandling;

  /**
   * Whether the client normalizes line endings to the client specific
   * setting.
   * <p>
   * If set to {@code true} the client will normalize line ending characters
   * in a workspace edit to the client specific new line character(s).
   */
  @ProtocolSince("3.16.0")
  private Boolean normalizesLineEndings;

  /**
   * Whether the client in general supports change annotations on text edits,
   * create file, rename file and delete file changes.
   */
  @ProtocolSince("3.16.0")
  private WorkspaceEditChangeAnnotationSupportCapabilities changeAnnotationSupport;

  /**
   * Whether the client supports snippets as text edits.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  private Boolean snippetEditSupport;

  /**
   * Whether the client supports {@link WorkspaceEditMetadata} in workspace edits.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  private Boolean metadataSupport;

  public WorkspaceEditCapabilities() {
  }

  /**
   * The client supports versioned document changes in {@link WorkspaceEdit}s
   */
  public Boolean getDocumentChanges() {
    return this.documentChanges;
  }

  /**
   * The client supports versioned document changes in {@link WorkspaceEdit}s
   */
  public void setDocumentChanges(final Boolean documentChanges) {
    this.documentChanges = documentChanges;
  }

  /**
   * The resource operations the client supports. Clients should at least
   * support 'create', 'rename' and 'delete' files and folders.
   * <p>
   * See {@link ResourceOperationKind} for allowed values.
   */
  @ProtocolSince("3.13.0")
  public List<String> getResourceOperations() {
    return this.resourceOperations;
  }

  /**
   * The resource operations the client supports. Clients should at least
   * support 'create', 'rename' and 'delete' files and folders.
   * <p>
   * See {@link ResourceOperationKind} for allowed values.
   */
  @ProtocolSince("3.13.0")
  public void setResourceOperations(final List<String> resourceOperations) {
    this.resourceOperations = resourceOperations;
  }

  /**
   * The failure handling strategy of a client if applying the workspace edit
   * fails.
   * <p>
   * See {@link FailureHandlingKind} for allowed values.
   */
  @ProtocolSince("3.13.0")
  public String getFailureHandling() {
    return this.failureHandling;
  }

  /**
   * The failure handling strategy of a client if applying the workspace edit
   * fails.
   * <p>
   * See {@link FailureHandlingKind} for allowed values.
   */
  @ProtocolSince("3.13.0")
  public void setFailureHandling(final String failureHandling) {
    this.failureHandling = failureHandling;
  }

  /**
   * Whether the client normalizes line endings to the client specific
   * setting.
   * <p>
   * If set to {@code true} the client will normalize line ending characters
   * in a workspace edit to the client specific new line character(s).
   */
  @ProtocolSince("3.16.0")
  public Boolean getNormalizesLineEndings() {
    return this.normalizesLineEndings;
  }

  /**
   * Whether the client normalizes line endings to the client specific
   * setting.
   * <p>
   * If set to {@code true} the client will normalize line ending characters
   * in a workspace edit to the client specific new line character(s).
   */
  @ProtocolSince("3.16.0")
  public void setNormalizesLineEndings(final Boolean normalizesLineEndings) {
    this.normalizesLineEndings = normalizesLineEndings;
  }

  /**
   * Whether the client in general supports change annotations on text edits,
   * create file, rename file and delete file changes.
   */
  @ProtocolSince("3.16.0")
  public WorkspaceEditChangeAnnotationSupportCapabilities getChangeAnnotationSupport() {
    return this.changeAnnotationSupport;
  }

  /**
   * Whether the client in general supports change annotations on text edits,
   * create file, rename file and delete file changes.
   */
  @ProtocolSince("3.16.0")
  public void setChangeAnnotationSupport(final WorkspaceEditChangeAnnotationSupportCapabilities changeAnnotationSupport) {
    this.changeAnnotationSupport = changeAnnotationSupport;
  }

  /**
   * Whether the client supports snippets as text edits.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  public Boolean getSnippetEditSupport() {
    return this.snippetEditSupport;
  }

  /**
   * Whether the client supports snippets as text edits.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  public void setSnippetEditSupport(final Boolean snippetEditSupport) {
    this.snippetEditSupport = snippetEditSupport;
  }

  /**
   * Whether the client supports {@link WorkspaceEditMetadata} in workspace edits.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  public Boolean getMetadataSupport() {
    return this.metadataSupport;
  }

  /**
   * Whether the client supports {@link WorkspaceEditMetadata} in workspace edits.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  public void setMetadataSupport(final Boolean metadataSupport) {
    this.metadataSupport = metadataSupport;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("documentChanges", this.documentChanges);
    b.add("resourceOperations", this.resourceOperations);
    b.add("failureHandling", this.failureHandling);
    b.add("normalizesLineEndings", this.normalizesLineEndings);
    b.add("changeAnnotationSupport", this.changeAnnotationSupport);
    b.add("snippetEditSupport", this.snippetEditSupport);
    b.add("metadataSupport", this.metadataSupport);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    WorkspaceEditCapabilities other = (WorkspaceEditCapabilities) obj;
    if (this.documentChanges == null) {
      if (other.documentChanges != null)
        return false;
    } else if (!this.documentChanges.equals(other.documentChanges))
      return false;
    if (this.resourceOperations == null) {
      if (other.resourceOperations != null)
        return false;
    } else if (!this.resourceOperations.equals(other.resourceOperations))
      return false;
    if (this.failureHandling == null) {
      if (other.failureHandling != null)
        return false;
    } else if (!this.failureHandling.equals(other.failureHandling))
      return false;
    if (this.normalizesLineEndings == null) {
      if (other.normalizesLineEndings != null)
        return false;
    } else if (!this.normalizesLineEndings.equals(other.normalizesLineEndings))
      return false;
    if (this.changeAnnotationSupport == null) {
      if (other.changeAnnotationSupport != null)
        return false;
    } else if (!this.changeAnnotationSupport.equals(other.changeAnnotationSupport))
      return false;
    if (this.snippetEditSupport == null) {
      if (other.snippetEditSupport != null)
        return false;
    } else if (!this.snippetEditSupport.equals(other.snippetEditSupport))
      return false;
    if (this.metadataSupport == null) {
      if (other.metadataSupport != null)
        return false;
    } else if (!this.metadataSupport.equals(other.metadataSupport))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.documentChanges== null) ? 0 : this.documentChanges.hashCode());
    result = prime * result + ((this.resourceOperations== null) ? 0 : this.resourceOperations.hashCode());
    result = prime * result + ((this.failureHandling== null) ? 0 : this.failureHandling.hashCode());
    result = prime * result + ((this.normalizesLineEndings== null) ? 0 : this.normalizesLineEndings.hashCode());
    result = prime * result + ((this.changeAnnotationSupport== null) ? 0 : this.changeAnnotationSupport.hashCode());
    result = prime * result + ((this.snippetEditSupport== null) ? 0 : this.snippetEditSupport.hashCode());
    return prime * result + ((this.metadataSupport== null) ? 0 : this.metadataSupport.hashCode());
  }
}
