/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.scheduler.product;

import static org.eclipse.lsat.scheduler.product.ProductUtil.getAction;
import static org.eclipse.lsat.scheduler.product.ProductUtil.isSuccessor;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.stream.Stream;

import org.eclipse.emf.common.util.EMap;
import org.eclipse.lsat.common.scheduler.graph.Task;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledTask;
import org.eclipse.lsat.product.productdata.LocationInformation;
import org.eclipse.lsat.product.productdata.ProductProperty;
import org.eclipse.lsat.product.productdata.ProductProperty.WHEN;

import activity.ActivityFactory;
import activity.Move;
import product.Property;
import product.TransferType;

public class ProductChange {
    /**
     *
     */
    private static final String PRODUCT = "product";

    @Override
    public String toString() {
        return "ProductChange [ scheduledTask=" + scheduledTask.getName() + ", activity="
                + scheduledTask.getTask().getGraph().getName() + ", (slot=" + slot + ", productOwner=" + productOwner
                + ", transferType=" + getTransferType() + "]";
    }

    private final ScheduledTask<Task> scheduledTask;

    private final activity.ProductChange activityProductChange;

    private final ProductOwner productOwner;

    private final TransferType transferType;

    private final org.eclipse.lsat.product.productdata.Task taskData;

    private final String slot;

    public ProductChange(ScheduledTask<Task> scheduledTask, org.eclipse.lsat.product.productdata.Task taskData,
            activity.ProductChange activityProductChange)
    {
        this.scheduledTask = scheduledTask;
        this.taskData = taskData;
        if (activityProductChange == null) {
            activityProductChange = ActivityFactory.eINSTANCE.createProductChange();
            getAction(scheduledTask).getProductChanges().add(activityProductChange);
        }
        this.activityProductChange = activityProductChange;
        this.transferType = activityProductChange.getTransferType();
        this.slot = activityProductChange.getSlot();
        this.productOwner = new ProductOwner(getAction(scheduledTask));
    }

    public void initProduct(String productId) {
        setStatus(productId, null);
    }

    public void setStatus(String suppliedProductId, ProductChange predecessor) {
        if (suppliedProductId == null && predecessor == null) {
            return;
        }
        var productId = suppliedProductId == null ? "?" : suppliedProductId;
        var map = this.scheduledTask.getProperties();
        if (predecessor != null) {
            for (var e: predecessor.scheduledTask.getProperties().entrySet()) {
                map.put(e.getKey(), e.getValue());
            }
        }
        var product = activityProductChange.getProduct();
        if (product != null) {
            putConditional(map, productId, "name", getStartTime(), product.getName());
        }
        if (transferType == TransferType.EXIT) {
            putConditional(map, productId, "exit", getStartTime(), "true");
        }
        putConditional(map, productId, "id", getStartTime(), productId);
        putConditional(map, productId, "slot", getStartTime(), getSlot());

        for (var p: activityProductChange.getStartProperties()) {
            putConditional(map, productId, p.getDefinition().getName(), getStartTime(),
                    p.getValue().getValueAsSerializable());
        }
        for (var p: activityProductChange.getEndProperties()) {
            putConditional(map, productId, p.getDefinition().getName(), getEndTime(),
                    p.getValue().getValueAsSerializable());
        }
    }

    public String getSlot() {
        return slot;
    }

    public boolean isTransfer() {
        return getTransferType() == TransferType.IN || getTransferType() == TransferType.OUT;
    }

    public boolean isEntry() {
        return getTransferType() == TransferType.ENTRY;
    }

    public boolean isExit() {
        return getTransferType() == TransferType.EXIT;
    }

    public boolean isMove() {
        return getAction(scheduledTask) instanceof Move;
    }

    public ProductOwner getProductOwner() {
        return productOwner;
    }

    public List<Property> getStartProperties() {
        return activityProductChange.getStartProperties();
    }

    public List<Property> getEndProperties() {
        return activityProductChange.getEndProperties();
    }

    public BigDecimal getStartTime() {
        return scheduledTask.getStartTime();
    }

    public BigDecimal getEndTime() {
        return scheduledTask.getEndTime();
    }

    public TransferType getTransferType() {
        return transferType;
    }

    public org.eclipse.lsat.product.productdata.Task getTaskData() {
        return taskData;
    }

    public boolean isMatch(ProductChange target) {
        // only a counterTransfer (in for out or out for in) is a match
        if (!target.isTransfer() || getTransferType().equals(target.getTransferType())) {
            return false;
        }

        if (getProductOwner().equals(target.getProductOwner())) {
            return false;
        }

        if (!Objects.equals(getSlot(), target.getSlot())) {
            return false;
        }

        return isSuccessor(scheduledTask.getTask(), target.scheduledTask.getTask()); // if no relation then no match
    }

    public List<ProductProperty> getChangedProperties() {
        return Stream.concat(
                getStartProperties().stream()
                        .map(prop -> new ProductProperty(taskData, WHEN.start, prop.getDefinition().getName(),
                                prop.getValue().getValueAsSerializable())),

                getEndProperties().stream().map(prop -> new ProductProperty(taskData, WHEN.end,
                        prop.getDefinition().getName(), prop.getValue().getValueAsSerializable())))
                .toList();
    }

    public LocationInformation createLocationInformation() {
        var slot = getSlot();
        return new LocationInformation(taskData, slot);
    }

    private static void putConditional(EMap<String, String> map, String productId, String key, BigDecimal since,
            Serializable value)
    {
        var fullKey = PRODUCT + "/" + productId + "/" + key;
        putConditional(map, fullKey, since, value);
    }

    private static void putConditional(EMap<String, String> map, String fullKey, BigDecimal since, Serializable value) {
        if (value != null && !value.toString().isBlank()) {
            var preceding = map.get(fullKey);
            var pValue = value.toString() + " (since: " + since + ")";
            if (preceding != null) {
                var precedingValue = preceding.toString().replaceFirst("(.*) \\(since: .*\\)", "$1");
                if (precedingValue.equals(value)) {
                    pValue = preceding.toString();
                }
            }
            map.put(fullKey, pValue);
        }
    }
}
